/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F4011"  Board */
/* Hardware    : ET-BASE dsPIC30F4011                 */
/* Target MCU  : dsPIC30F4011                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Use I2C Connect PCF8583      */
/*             : Display Clock Time ---> UART         */	
/******************************************************/
/* Used ET-MINI I/O RTC:PCF8583 Module Interface      */
/* SDA = RF2(I2C-SDA)                                 */
/* SCL = RF3(I2C-SCL)                                 */
/* Set Address PCF8583 = 1010001x                     */
/******************************************************/

#include <p30f4011.h>                                		// dsPIC30F4011 MPU Register
#include <stdio.h>											// Used "sprintf" Function
#include "uart.h"											// Used UART Config
#include "i2c.h"											// Used I2C Function Library

/* Setup Configuration For ET-BASE dsPIC30F4011 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);								// Disable Clock Switching,Enable Fail-Salf Clock
                                                            // Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					    // Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End Configuration For ET-BASE dsPIC30F4011 */

char uart_buf[40]; 											// "sprintf" Buffer

/* pototype  section */
void init_uart(void);										// Initial UART = 9600,N,8,1
void init_i2c(void);										// Initial I2C For Interface PCF8583
unsigned char ReadPCF8583(unsigned char Read_Addr);			// Read RTC:PCF8583 Time
void WritePCF8583(unsigned char Write_Addr,					// Write RTC:PCF8583 Time
                unsigned char Set_Time);		


int main(void)
{  
  unsigned char Hour,Minute,Second; 						// RTC:PCF8583 Buffer Data
  unsigned char Last_Second;  
  init_uart();												// Initial UART = 9600,N,8,1
  init_i2c();												// Initial I2C Function
   
  sprintf(uart_buf,"\n\n\rET-BASE dsPIC30F4011 : I2C\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete
  
  sprintf(uart_buf,"Demo Interface I2C RTC:PCF8583\n\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  

  //**************************************//
  // Start Setup RTC : PCF8583 = 00:00:00 //
  //**************************************//
  WritePCF8583(0x04,0x00);									// Setup Hour = 00	
  WritePCF8583(0x03,0x00);									// Setup Minute = 00
  WritePCF8583(0x02,0x00);									// Setup Second = 00
  WritePCF8583(0x00,0x00);									// Setup PCF8583 = RTC Function
  Last_Second = 59;											// Default Second

  //********************************//
  // Start Read RTC Clock & Display //
  //********************************//
  while(1)
  { 
    do 														// Repeat Get Second until Second Change
	{
	  Hour   = ReadPCF8583(0x04);							// Read Hour (00-23)
	  Minute = ReadPCF8583(0x03);							// Read Minute (00-59)
      Second = ReadPCF8583(0x02);							// Read Second (00-59)
    }
    while(Last_Second == Second);							// Repeat if Second Not Change
	
    Last_Second = Second;									// Update Current Second
          
    //************************************//
    // Display Clock = Hour:Minute:Second //
    //************************************//
    sprintf(uart_buf,"\rPCF8583 Clock = ");					// Print Message String 
  	putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
  	while(BusyUART1());										// Wait putsUART1 Complete
   
	WriteUART1(((Hour>>4)&0x03)|0x30);			
    while(BusyUART1());										// Wait putsUART1 Complete   
    WriteUART1((Hour&0x0F)|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete

	WriteUART1(':');	
    while(BusyUART1());										// Wait putsUART1 Complete
    
	WriteUART1(((Minute>>4)&0x07)|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete    
    WriteUART1((Minute&0x0F)|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete

    WriteUART1(':');
    while(BusyUART1());										// Wait putsUART1 Complete	
   
	WriteUART1(((Second>>4)&0x07)|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete    
    WriteUART1((Second&0x0F)|0x30);
    while(BusyUART1());										// Wait putsUART1 Complete    
  }
}

/*********************************/
/* Initial UART for dsPIC30F4011 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart()
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  
  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
  			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-BASE dsPIC30F4011 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-BASE dsPIC30F4011 UART Baudrate = 9600 BPS
}

/*************************************/
/* Initial dsPIC30F4011 I2C Function */
/* For Interface PCF8583 (I2C RTC)   */
/*************************************/
void init_i2c(void)
{  
  CloseI2C();												// Close I2C Before New Config  

  // Open I2C Function For Interface PCF8583
  OpenI2C(I2C_ON &											// Enable I2C Function
		  I2C_IDLE_STOP &									// Disable I2C in IDLE Mode
		  I2C_CLK_HLD &										// I2C Clock Hold
		  I2C_IPMI_DIS &     								// Disable I2C IPMI Mode Control
		  I2C_7BIT_ADD &									// I2C Device Address = 7 Bit
		  I2C_SLW_DIS &										// Disable I2C Slew Rate Control
		  I2C_SM_DIS &										// Disable I2C SMBUS Mode
		  I2C_GCALL_DIS &									// Disable I2C General Call(Slave) 
          I2C_STR_DIS &										// Disable SCL Clock Stretch
          I2C_ACK &											// ACK Cycle = ACK
	      I2C_ACK_DIS &										// Disable I2C Acknowledge
		  I2C_RCV_DIS &										// Disable I2C Receive
		  I2C_STOP_DIS &									// Disable I2C Stop		
		  I2C_RESTART_DIS &									// Disable I2C Restart
		  I2C_START_DIS,									// Disable I2C Start		

          // ET-BASE dsPIC30F4011 Hardware Board
		  // XTAL = 7.3728MHz
  		  // Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  		  // Fcy(I2C) = Fosc / 4 
  		  //           = 117.9648 / 4 = 29.4912 MHz          
  		  // I2CBRG = [(Fcy/Fscl)-(Fcy/1,111,111)]-1
  		  //       = [(29.4912 MHz / 100KHz)-(29.4912MHz / 1,111,111)] - 1
          //       = [(294.912)-(26.542)]-1
          //       = 268.37 - 1
  		  //       = 267 = 10BH
		  267);												// I2C Baudrate(Approx. = 100 KHz)

  // Initial I2C Interrupt Control
  ConfigIntI2C(MI2C_INT_OFF &								// Disabe Master I2C Interrupt
               SI2C_INT_OFF &								// Disabe Slave I2C Interrupt
               MI2C_INT_PRI_7 &								// Set Priority Interrupt of Master = 7 
			   SI2C_INT_PRI_7 );							// Set Priority Interrupt of Slave = 7                 
}

/*************************/
/* Read RTC:PCF8583 Time */
/*************************/
unsigned char ReadPCF8583(unsigned char Read_Addr)			// Read RTC:PCF8583 Time
{
  unsigned char Get_Time;									// Time Buffer

  StartI2C();												// Send Start Condition
  while(I2CCONbits.SEN);									// Wait Start Complete
    
  // Write PCF8583 ID Code = 1010001+W
  MasterWriteI2C(0xA2);										// Write PCF8583 ID Code,Write
  while(I2CSTATbits.TBF);									// Wait Write ID Code Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  

  // Write Address of RTC:PCF8583 For Read
  MasterWriteI2C(Read_Addr);								// Write RTC Address 
  while(I2CSTATbits.TBF);									// Wait Write Address Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  
       
  // Restart For Read PCF8583 Data
  RestartI2C();												// Send Re-Start Condition
  while(I2CCONbits.RSEN);									// Wait Re-Start Complete
     
  // Write PCF8583 ID Code = 1010001+R 
  MasterWriteI2C(0xA3);										// Write PCF8583 ID Code,Read
  while(I2CSTATbits.TBF);									// Wait Write ID Code Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  
    
  Get_Time = MasterReadI2C();								// Read Time From RTC
  while(I2CSTATbits.RBF);									// Wait Read Data Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear 
        
  StopI2C();												// Send Stop Condition
  while(I2CCONbits.PEN);									// Wait Stop Complete    

  return Get_Time;											// Return Time Result
}

/**************************/
/* Write RTC:PCF8583 Time */
/**************************/
void WritePCF8583(unsigned char Write_Addr,					// Write RTC:PCF8583 Time
                unsigned char Set_Time)						
{
  StartI2C();												// Send Start Condition
  while(I2CCONbits.SEN);									// Wait Start Complete

  // Write PCF8583 ID Code = 1010001+W  
  MasterWriteI2C(0xA2);										// Write PCF8583 ID Code (1010001+W)
  while(I2CSTATbits.TBF);									// Wait Write ID Code Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear     
         
  // Write RTC Address
  MasterWriteI2C(Write_Addr);								// Write RTC Address
  while(I2CSTATbits.TBF);									// Wait Write Address Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  

  // Write Time to RTC:PCF8583   
  MasterWriteI2C(Set_Time);									// Write Time to RTC:PCF8583
  while(I2CSTATbits.TBF);									// Wait Write Time Complete
  while(I2CSTATbits.ACKSTAT);								// Wait Acknowledge Complete
  IdleI2C();												// Wait Status Clear  
  
  StopI2C();												// Send Stop Condition
  while(I2CCONbits.PEN);									// Wait Stop Complete
}
